import { world, system, Player } from "@minecraft/server";
import { ActionFormData, ModalFormData, MessageFormData } from "@minecraft/server-ui";
import { cachedPlayers } from './lag/pvp.js';

const protectedRegions = [
  {
    min: { x: 3943, y: 5, z: -4054 }, // mining sim auto gen areas
    max: { x: 3945, y: 11, z: -4056 }
  },
  {
    min: { x: 3943, y: 5, z: -4064 }, // mining sim auto gen areas
    max: { x: 3945, y: 11, z: -4062 }
  },
  {
    min: { x: 3943, y: 5, z: -4072 }, // mining sim auto gen areas
    max: { x: 3945, y: 11, z: -4070 }
  },
  {
    min: { x: 3943, y: 5, z: -4078 }, // mining sim auto gen areas
    max: { x: 3945, y: 11, z: -4080 }
  },
  {
    min: { x: 3933, y: 5, z: -4078 }, // mining sim auto gen areas
    max: { x: 3935, y: 11, z: -4080 }
  },
  {
    min: { x: 3933, y: 5, z: -4072 }, // mining sim auto gen areas
    max: { x: 3935, y: 11, z: -4070 }
  },
  {
    min: { x: 3933, y: 5, z: -4064 }, // mining sim auto gen areas
    max: { x: 3935, y: 11, z: -4062 }
  },
  {
    min: { x: 3933, y: 5, z: -4056 }, // mining sim auto gen areas
    max: { x: 3935, y: 11, z: -4054 }
  },
  {
    min: { x: 3929, y: 5, z: -4048 }, // mining sim auto gen areas
    max: { x: 3931, y: 11, z: -4046 }
  },
  {
    min: { x: 3957, y: 5, z: -4048 }, // mining sim auto gen areas
    max: { x: 3959, y: 11, z: -4046 }
  },
  {
    min: { x: 3953, y: 5, z: -4056 }, // mining sim auto gen areas
    max: { x: 3955, y: 11, z: -4054 }
  },
  {
    min: { x: 3953, y: 5, z: -4064 }, // mining sim auto gen areas
    max: { x: 3955, y: 11, z: -4062 }
  },
  {
    min: { x: 3953, y: 5, z: -4070 }, // mining sim auto gen areas
    max: { x: 3955, y: 11, z: -4072 }
  },
  {
    min: { x: 3953, y: 5, z: -4078 }, // mining sim auto gen areas
    max: { x: 3955, y: 11, z: -4080 }
  },
  {
    min: { x: 3003, y: 6, z: 3001 }, // Admin shop
    max: { x: 3004, y: 10, z: 3001 }
  },
  {
    min: { x: -3971, y: 12, z: -3891 }, // vip Island 
    max: { x: -3985, y: 15, z: -3877 }
  },
  {
    min: { x: -3971, y: 17, z: -3891 }, // vip Island 
    max: { x: -3972, y: 22, z: -3877 }
  },
  {
    min: { x: -3973, y: 17, z: -3891 }, // vip Island 
    max: { x: -3985, y: 22, z: -3890 }
  },
  {
    min: { x: -3985, y: 17, z: -3889 }, // vip Island 
    max: { x: -3984, y: 22, z: -3877 }
  },
  {
    min: { x: -3983, y: 17, z: -3877 }, // vip Island 
    max: { x: -3973, y: 20, z: -3878 }
  },
  {
    min: { x: -3983, y: 19, z: -3889 }, // vip Island 
    max: { x: -3973, y: 22, z: -3879 }
  },
  {
    min: { x: 2977, y: 47, z: -3037 },
    max: { x: 2993, y: 26, z: -3083 }
  },
  {
    min: { x: 3023, y: 47, z: -3026 },
    max: { x: 3019, y: 26, z: -3074 }
  },
  {
    min: { x: 2977, y: 47, z: -2982 },
    max: { x: 2990, y: 26, z: -2917 }
  },
  {
    min: { x: 3959, y: 4, z: -4015 },
    max: { x: 3929, y: 11, z: -3985 }
  },
  {
    min: { x: 2976, y: 4, z: -2988 },
    max: { x: 2994, y: 12, z: -2916 }
  },
  {
    min: { x: 3024, y: 4, z: -2925 },
    max: { x: 3006, y: 12, z: -2979 }
  },
  {
    min: { x: 2976, y: 4, z: -3012 },
    max: { x: 2994, y: 12, z: -3084 }
  },
  {
    min: { x: 3006, y: 4, z: -3075 },
    max: { x: 3024, y: 12, z: -3021 }
  }
];

function isInProtectedRegion(loc) {
  return protectedRegions.some(region =>
    loc.x >= Math.min(region.min.x, region.max.x) &&
    loc.x <= Math.max(region.min.x, region.max.x) &&
    loc.y >= Math.min(region.min.y, region.max.y) &&
    loc.y <= Math.max(region.min.y, region.max.y) &&
    loc.z >= Math.min(region.min.z, region.max.z) &&
    loc.z <= Math.max(region.min.z, region.max.z)
  );
}

world.beforeEvents.playerBreakBlock.subscribe(event => {
  const { block, player, itemStack: item } = event;
  if (!player || !block || !item) return;

  // Prevent breaking blocks in protected regions
  if (isInProtectedRegion(block.location)) {
    event.cancel = true;
    return;
  }

  // Prevent breaking blocks with a compass (for all players)
  if (item.typeId === "minecraft:compass") {
    event.cancel = true;
  }
});

world.afterEvents.playerPlaceBlock.subscribe(event => {
  const { block, player } = event;

  if (isInProtectedRegion(block.location)) {
    // Break the placed block by setting it to air
    block.dimension.runCommandAsync(`fill ${block.location.x} ${block.location.y} ${block.location.z} ${block.location.x} ${block.location.y} ${block.location.z} air destroy`);
  }
});




// === CONFIG ===
const startX = -3983;
const startZ = -3889;
const y = 18;

const PROXIMITY_CENTER = { x: -3978, y: 19, z: -3884 };
const PROXIMITY_RADIUS_SQ = 20 * 20;

const ORE_BLOCKS = [
  "coal_block",
  "iron_block",
  "copper_block",
  "lapis_block",
  "gold_block",
  "redstone_block",
  "diamond_block",
  "emerald_block",
  "netherite_block",
  "quartz_block"
];


function isPlayerNearby() {
  for (const player of cachedPlayers) {
    if (!player || !player.location) continue; // Defensive check
    const loc = player.location;
    const dx = loc.x - PROXIMITY_CENTER.x;
    const dy = loc.y - PROXIMITY_CENTER.y;
    const dz = loc.z - PROXIMITY_CENTER.z;
    if ((dx * dx + dy * dy + dz * dz) <= PROXIMITY_RADIUS_SQ) return true;
  }
  return false;
}

function fillOrePlatform() {
  const overworld = world.getDimension("overworld");
  let oreIndex = 0;

  for (let x = 0; x < 11; x += 2) {
    for (let z = 0; z < 11; z += 2) {
      const useThreeWide = (x % 4 === 0 && z % 4 === 0); // sprinkle a 3×2 every few chunks
      const blockType = ORE_BLOCKS[oreIndex % ORE_BLOCKS.length];
      oreIndex++;

      const x1 = startX + x;
      const z1 = startZ + z;
      const x2 = Math.min(x1 + (useThreeWide ? 2 : 1), startX + 10); // 3 or 2 blocks wide
      const z2 = Math.min(z1 + 1, startZ + 10); // always 2 blocks deep

      // catch any error from the async call, no await needed here
      overworld.runCommandAsync(`fill ${x1} ${y} ${z1} ${x2} ${y} ${z2} ${blockType}`)
        .catch(e => console.warn(`❌ Fill failed at (${x1}, ${y}, ${z1}): ${e}`));
    }
  }
}

// Run every tick, fill only if player is nearby
system.runInterval(() => {
  if (isPlayerNearby()) {
    fillOrePlatform();
  }
}, 1);












//vip button


function removeItems(player, typeId, amount) {
    const inventory = player.getComponent("minecraft:inventory").container;
    for (let slot = 0; slot < inventory.size; slot++) {
        const item = inventory.getItem(slot);
        if (!item || item.typeId !== typeId) continue;

        if (item.amount <= amount) {
            amount -= item.amount;
            inventory.setItem(slot, undefined);
        } else {
            const reducedItem = item.clone();
            reducedItem.amount = item.amount - amount;
            inventory.setItem(slot, reducedItem);
            amount = 0;
        }

        if (amount <= 0) break;
    }
}

export function showBlockMenu(player) {
    const inventory = player.getComponent("minecraft:inventory").container;
    const moneyObjective = world.scoreboard.getObjective("Money");

    const BLOCKS = [
        { type: "minecraft:coal_block", name: "Coal Block", displayName: "§l§8Coal Block", price: 400 },
        { type: "minecraft:lapis_block", name: "Lapis Block", displayName: "§l§9Lapis Block", price: 150 },
        { type: "minecraft:iron_block", name: "Iron Block", displayName: "§l§fIron Block", price: 700 },
        { type: "minecraft:copper_block", name: "Copper Block", displayName: "§l§6Copper Block", price: 600 },
        { type: "minecraft:gold_block", name: "Gold Block", displayName: "§l§eGold Block", price: 800 },
        { type: "minecraft:diamond_block", name: "Diamond Block", displayName: "§l§bDiamond Block", price: 1000 },
        { type: "minecraft:emerald_block", name: "Emerald Block", displayName: "§l§aEmerald Block", price: 900 },
        { type: "minecraft:redstone_block", name: "Redstone Block", displayName: "§l§cRedstone Block", price: 300 },
        { type: "minecraft:quartz_block", name: "Quartz Block", displayName: "§l§fQuartz Block", price: 250 },
        { type: "minecraft:netherite_block", name: "Netherite Block", displayName: "§l§7Netherite Block", price: 8000 }
    ];

    const inventoryCounts = {};
    let totalSellable = 0;
    for (let slot = 0; slot < inventory.size; slot++) {
        const item = inventory.getItem(slot);
        if (!item) continue;
        inventoryCounts[item.typeId] = (inventoryCounts[item.typeId] || 0) + item.amount;
    }
    for (const block of BLOCKS) {
        totalSellable += inventoryCounts[block.type] ?? 0;
    }

    if (totalSellable === 0) {
        player.sendMessage("§cYou don't have any sellable blocks.");
        player.playSound("note.bass");
        return;
    }

   
    const form = new ModalFormData()
        .title("§lVIP Minerals Sell")
        .dropdown("§7Select a mode", [
            "§aManual Sell (Use sliders)",
            `§cSell All (${totalSellable})`
        ]);
        
    const priceInfo = BLOCKS.map(b => `§7- ${b.displayName} §f= §a$${b.price}`).join("\n");
    
    form.slider(
        `§l§6Welcome to VIP Minerals Sell!\n§r§fUse the sliders below to sell specific amounts,\nor choose §c"Sell All"§f above to sell everything.\n\n§l§eSellable Blocks:\n${priceInfo}`,
        0, 0, 0, 0
    );

    for (const block of BLOCKS) {
        const count = inventoryCounts[block.type] ?? 0;
        form.slider(`${block.displayName} §7(0–${count})`, 0, count, 1, 0);
    }

    form.show(player).then(res => {
        if (res.canceled) return;

        const selectedMode = res.formValues[0];
        const sliderValues = res.formValues.slice(2);

        if (selectedMode === 1) {
            // SELL ALL MODE
            let totalAmount = 0;
            let totalMoney = 0;
            let summaryLines = [];

            for (const block of BLOCKS) {
                const count = inventoryCounts[block.type] ?? 0;
                if (count > 0) {
                    const earn = count * block.price;
                    totalAmount += count;
                    totalMoney += earn;
                    summaryLines.push(`${block.displayName} §rx §e${count} §7= §a$${earn}§r`);
                }
            }

            const confirm = new MessageFormData()
                .title("§lConfirm Sell All")
                .body(`You're selling:\n\n${summaryLines.join("\n")}\n\n§l§fTotal: §a§l$${totalMoney}§r`)
                .button1("§cCancel")
                .button2("§aConfirm");

            confirm.show(player).then(confirmRes => {
                if (confirmRes.selection !== 1) {
                    player.sendMessage("§cSell cancelled.");
                    player.playSound("note.bass");
                    return;
                }

                for (const block of BLOCKS) {
                    const count = inventoryCounts[block.type] ?? 0;
                    if (count > 0) removeItems(player, block.type, count);
                }

                if (moneyObjective) {
                    const current = moneyObjective.getScore(player) ?? 0;
                    moneyObjective.setScore(player, current + totalMoney);
                }

                player.sendMessage(`§gYou sold ${totalAmount} blocks for §a$${totalMoney}`);
                player.playSound("random.orb");
            });

        } else {
            // MANUAL SLIDER MODE
            let totalAmount = 0;
            let totalMoney = 0;
            let summaryLines = [];

            for (let i = 0; i < sliderValues.length; i++) {
                const amount = sliderValues[i];
                const block = BLOCKS[i];
                if (amount > 0) {
                    const earn = amount * block.price;
                    totalAmount += amount;
                    totalMoney += earn;
                    summaryLines.push(`${block.displayName} §rx §e${amount} §7= §a$${earn}§r`);
                }
            }

            if (totalAmount === 0) {
                player.sendMessage("§cNothing selected to sell!");
                player.playSound("note.bass");
                return;
            }

            const confirm = new MessageFormData()
                .title("§lConfirm Sale")
                .body(`You're selling:\n\n${summaryLines.join("\n")}\n\n§l§fTotal: §a§l$${totalMoney}§r`)
                .button1("§cCancel")
                .button2("§aConfirm");

            confirm.show(player).then(confirmRes => {
                if (confirmRes.selection !== 1) {
                    player.sendMessage("§cSell cancelled.");
                    player.playSound("note.bass");
                    return;
                }

                for (let i = 0; i < sliderValues.length; i++) {
                    const amountToRemove = sliderValues[i];
                    if (amountToRemove > 0) {
                        removeItems(player, BLOCKS[i].type, amountToRemove);
                    }
                }

                if (moneyObjective) {
                    const current = moneyObjective.getScore(player) ?? 0;
                    moneyObjective.setScore(player, current + totalMoney);
                }

                player.sendMessage(`§gYou sold ${totalAmount} blocks for §a$${totalMoney}`);
                player.playSound("random.orb");
            });
        }
    });
}

